<?php
session_start();

// Require role
if (!isset($_SESSION['role'])) {
    header('Location: unauthorized.php');
    exit();
}

require_once 'db_connection.php'; // provides $conn (mysqli), utf8mb4

$role          = $_SESSION['role'];
$user_id       = isset($_SESSION['user_id']) ? (int)$_SESSION['user_id'] : 0;
$user_branchId = isset($_SESSION['branch_id']) ? (int)$_SESSION['branch_id'] : 0;

// --- Read filters (POST) safely ---
$start_date = isset($_POST['start_date']) ? trim($_POST['start_date']) : '';
$end_date   = isset($_POST['end_date'])   ? trim($_POST['end_date'])   : '';
$branch_id  = isset($_POST['branch_id'])  ? (int)$_POST['branch_id']   : 0;
$priority   = isset($_POST['priority'])   ? trim($_POST['priority'])   : '';

// Enforce role-based scope:
// - staff_user: only letters.forward_to = current user
// - branch_head: only own branch
// - letter_admin / super_admin: can see all (and can filter by branch_id)
$conditions = ["letters.action = 'yes'"];
$params     = [];
$types      = "";

// staff_user scope
if ($role === 'staff_user') {
    $conditions[] = "letters.forward_to = ?";
    $params[] = $user_id;
    $types .= "i";
}

// branch_head scope
if ($role === 'branch_head') {
    $conditions[] = "letters.branch_id = ?";
    $params[] = $user_branchId;
    $types .= "i";
}

// Admin-level optional branch filter (only if user is admin/super_admin)
$adminLike = ($role === 'letter_admin' || $role === 'super_admin');
if ($adminLike && $branch_id > 0) {
    $conditions[] = "letters.branch_id = ?";
    $params[] = $branch_id;
    $types .= "i";
}

// Date range filter (if both provided)
if ($start_date !== '' && $end_date !== '') {
    // Basic format check (YYYY-MM-DD); if invalid, ignore silently
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $start_date) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $end_date)) {
        $conditions[] = "letters.letter_date BETWEEN ? AND ?";
        $params[] = $start_date; $params[] = $end_date;
        $types .= "ss";
    }
}

// Priority filter
if ($priority !== '') {
    $conditions[] = "letters.priority = ?";
    $params[] = $priority;
    $types .= "s";
}

$where = $conditions ? "WHERE " . implode(" AND ", $conditions) : "";

// Main query
$sql = "
SELECT
  letters.letter_id,
  letters.letter_number,
  letters.letter_date,
  letters.department,
  letters.subject,
  branches.branch_name,
  letters.scan_copy,
  letters.forward_to,
  users.username AS forward_to_name,
  letters.description,
  letters.action,
  letters.hod_comment,
  letters.priority
FROM letters
JOIN branches ON letters.branch_id = branches.branch_id
LEFT JOIN users ON letters.forward_to = users.user_id
{$where}
ORDER BY letters.letter_date DESC, letters.letter_id DESC
";

$stmt = $conn->prepare($sql);
if ($types !== "") {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();

// Branch list only for admin-like users
$branches = null;
if ($adminLike) {
    $branches = $conn->query("SELECT branch_id, branch_name FROM branches ORDER BY branch_name ASC");
}
$priorities = ['High', 'Medium', 'Low'];

// Keep connection open if needed later
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Letters — Action Taken</title>

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
        rel="stylesheet"
        integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
        crossorigin="anonymous"/>

  <!-- DataTables (Bootstrap 5 theme) -->
  <link rel="stylesheet"
        href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css"/>

  <!-- Icons -->
  <link rel="stylesheet"
        href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"/>

  <style>
    :root { --brand:#0d6efd; --bg:#f6f8fb; }
    body { background: var(--bg); }
    .page-head {
      background: linear-gradient(135deg, var(--brand), #5aa2ff);
      color: #fff; border-radius: 1rem; padding: 1rem 1.25rem;
      box-shadow: 0 6px 24px rgba(13,110,253,.25);
    }
    .card-filter { border: none; border-radius: 1rem; }
    .table thead th { white-space: nowrap; }
    .high-priority { background-color: #ffe1e1; } /* softer red */
    .btn-reset { white-space: nowrap; }
  </style>
</head>
<body>
<div class="container-fluid py-3">

  <!-- Header -->
  <div class="page-head mb-3">
    <div class="d-flex align-items-center justify-content-between">
      <div class="d-flex align-items-center gap-3">
        <i class="fa-solid fa-circle-check fs-3"></i>
        <div>
          <h5 class="mb-0 fw-bold">Letters — Action Taken by Staff</h5>
          <small>Showing letters where <code>action = 'yes'</code></small>
        </div>
      </div>
      <div class="d-flex gap-2">
        <?php if ($role === 'letter_admin'): ?>
          <a class="btn btn-light" href="letter_admin_dashboard.php">
            <i class="fa-solid fa-arrow-left-long me-1"></i> Dashboard
          </a>
        <?php elseif ($role === 'branch_head'): ?>
          <a class="btn btn-light" href="branch_head_dashboard.php">
            <i class="fa-solid fa-arrow-left-long me-1"></i> Dashboard
          </a>
        <?php elseif ($role === 'super_admin'): ?>
          <a class="btn btn-light" href="super_admin_dashboard.php">
            <i class="fa-solid fa-arrow-left-long me-1"></i> Dashboard
          </a>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <!-- Filters -->
  <div class="card card-filter shadow-sm mb-3">
    <div class="card-body">
      <form action="action_yes_letters.php" method="POST" class="row g-3 align-items-end">

        <div class="col-12 col-md-3">
          <label class="form-label" for="start_date">Start Date</label>
          <input type="date" id="start_date" name="start_date" class="form-control"
                 value="<?= htmlspecialchars($start_date, ENT_QUOTES, 'UTF-8'); ?>">
        </div>

        <div class="col-12 col-md-3">
          <label class="form-label" for="end_date">End Date</label>
          <input type="date" id="end_date" name="end_date" class="form-control"
                 value="<?= htmlspecialchars($end_date, ENT_QUOTES, 'UTF-8'); ?>">
        </div>

        <?php if ($adminLike): ?>
          <div class="col-12 col-md-3">
            <label class="form-label" for="branch_id">Branch</label>
            <select id="branch_id" name="branch_id" class="form-select">
              <option value="">All Branches</option>
              <?php if ($branches && $branches->num_rows > 0): ?>
                <?php while ($b = $branches->fetch_assoc()): ?>
                  <option value="<?= (int)$b['branch_id']; ?>"
                    <?= ($branch_id === (int)$b['branch_id']) ? 'selected' : '' ?>>
                    <?= htmlspecialchars($b['branch_name'], ENT_QUOTES, 'UTF-8'); ?>
                  </option>
                <?php endwhile; ?>
              <?php endif; ?>
            </select>
          </div>
        <?php endif; ?>

        <div class="col-12 col-md-3">
          <label class="form-label" for="priority">Priority</label>
          <select id="priority" name="priority" class="form-select">
            <option value="">All Priorities</option>
            <?php foreach ($priorities as $p): ?>
              <option value="<?= htmlspecialchars($p, ENT_QUOTES, 'UTF-8'); ?>"
                <?= ($priority === $p ? 'selected' : '') ?>><?= htmlspecialchars($p, ENT_QUOTES, 'UTF-8'); ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-auto">
          <button type="submit" class="btn btn-primary">
            <i class="fa-solid fa-filter me-1"></i> Apply Filters
          </button>
        </div>
        <div class="col-auto">
          <a href="action_yes_letters.php" class="btn btn-outline-secondary btn-reset">
            <i class="fa-solid fa-rotate-left me-1"></i> Reset
          </a>
        </div>
      </form>
    </div>
  </div>

  <!-- Table -->
  <div class="card shadow-sm">
    <div class="table-responsive">
      <table id="lettersTable" class="table table-striped align-middle mb-0">
        <thead class="table-light">
        <tr>
          <th>Letter ID</th>
          <th>Letter No.</th>
          <th>Date</th>
          <th>Department</th>
          <th>Subject</th>
          <th>Branch</th>
          <th>Staff (Forward To)</th>
          <th>Description</th>
          <th>Scan</th>
          <th>HOD Comment</th>
          <th>Priority</th>
        </tr>
        </thead>
        <tbody>
        <?php if ($result && $result->num_rows > 0): ?>
          <?php while ($row = $result->fetch_assoc()): ?>
            <?php $isHigh = (isset($row['priority']) && $row['priority'] === 'High'); ?>
            <tr class="<?= $isHigh ? 'high-priority' : '' ?>">
              <td><?= (int)$row['letter_id']; ?></td>
              <td><?= htmlspecialchars($row['letter_number'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['letter_date'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['department'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['subject'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['branch_name'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['forward_to_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['description'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
              <td>
                <?php if (!empty($row['scan_copy'])): ?>
                  <a class="btn btn-sm btn-outline-primary"
                     href="<?= htmlspecialchars($row['scan_copy'], ENT_QUOTES, 'UTF-8'); ?>"
                     target="_blank"><i class="fa-solid fa-up-right-from-square me-1"></i>View</a>
                <?php else: ?>
                  <span class="text-muted">N/A</span>
                <?php endif; ?>
              </td>
              <td><?= htmlspecialchars($row['hod_comment'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['priority'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
            </tr>
          <?php endwhile; ?>
        <?php else: ?>
          <tr><td colspan="11" class="text-center py-4">No letters found.</td></tr>
        <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>

  <!-- Back button -->
  <div class="mt-3">
    <?php if ($role === 'letter_admin'): ?>
      <a href="letter_admin_dashboard.php" class="btn btn-light">
        <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
      </a>
    <?php elseif ($role === 'branch_head'): ?>
      <a href="branch_head_dashboard.php" class="btn btn-light">
        <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
      </a>
    <?php elseif ($role === 'super_admin'): ?>
      <a href="super_admin_dashboard.php" class="btn btn-light">
        <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
      </a>
    <?php endif; ?>
  </div>

</div>

<!-- JS: Bootstrap + DataTables (BS5) -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
<script>
  $(function () {
    $('#lettersTable').DataTable({
      pageLength: 25,
      order: [[2, 'desc']], // sort by Date desc
      columnDefs: [
        { targets: [0], visible: false }, // hide internal ID if you want
      ]
    });
  });
</script>
</body>
</html>
