<?php
session_start();

// Role gate
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'branch_head') {
    header('Location: unauthorized.php');
    exit();
}

require_once 'db_connection.php'; // mysqli, sets $conn + utf8mb4

$branchName = '—';
$username   = htmlspecialchars($_SESSION['username'] ?? 'User', ENT_QUOTES, 'UTF-8');
$branch_id  = isset($_SESSION['branch_id']) ? (int)$_SESSION['branch_id'] : 0;

/* Branch name */
if ($branch_id > 0) {
    $stmt = $conn->prepare("SELECT branch_name FROM branches WHERE branch_id = ?");
    $stmt->bind_param('i', $branch_id);
    $stmt->execute();
    $res = $stmt->get_result();
    if ($res && ($row = $res->fetch_assoc())) {
        $branchName = htmlspecialchars($row['branch_name'] ?? '—', ENT_QUOTES, 'UTF-8');
    }
    $stmt->close();
}

/* KPIs in a single query (includes two extra KPIs: overdue_no_action, today_received) */
$kpi = [
  'total' => 0,
  'pending' => 0,
  'received_not_assigned' => 0,
  'assigned' => 0,
  'not_seen' => 0,
  'seen_no_action' => 0,
  'action_yes' => 0,
  'action_no' => 0,
  'three_to_seven_no_action' => 0,
  'overdue_no_action' => 0,  // > 7 days with action = 'no'
  'today_received' => 0,     // DATE(received_date) = CURDATE()
];

if ($branch_id > 0) {
    $sql = "
      SELECT
        COUNT(*) AS total,
        SUM(CASE WHEN status_branch_head = 'pending' THEN 1 ELSE 0 END) AS pending,
        SUM(CASE WHEN status_branch_head = 'received' THEN 1 ELSE 0 END) AS received_not_assigned,
        SUM(CASE WHEN status_branch_head = 'received_assigned' THEN 1 ELSE 0 END) AS assigned,
        SUM(CASE WHEN forward_to IS NOT NULL AND forward_to <> 0 AND seen = 0 THEN 1 ELSE 0 END) AS not_seen,
        SUM(CASE WHEN forward_to IS NOT NULL AND forward_to <> 0 AND seen = 1 AND action = 'no' THEN 1 ELSE 0 END) AS seen_no_action,
        SUM(CASE WHEN action = 'yes' THEN 1 ELSE 0 END) AS action_yes,
        SUM(CASE WHEN action = 'no' THEN 1 ELSE 0 END) AS action_no,
        SUM(CASE WHEN action = 'no' AND DATEDIFF(CURDATE(), assign_date) BETWEEN 3 AND 7 THEN 1 ELSE 0 END) AS three_to_seven_no_action,
        SUM(CASE WHEN action = 'no' AND DATEDIFF(CURDATE(), assign_date) > 7 THEN 1 ELSE 0 END) AS overdue_no_action,
        SUM(CASE WHEN DATE(received_date) = CURDATE() THEN 1 ELSE 0 END) AS today_received
      FROM letters
      WHERE branch_id = ?
    ";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('i', $branch_id);
    $stmt->execute();
    $res = $stmt->get_result();
    if ($res && ($row = $res->fetch_assoc())) {
        foreach ($kpi as $k => $_) {
            $kpi[$k] = (int)($row[$k] ?? 0);
        }
    }
    $stmt->close();
}

// $conn->close(); // keep open if you fetch more later
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Branch Head Dashboard – Valvettithurai Urban Council</title>

  <!-- Bootstrap 5 -->
  <link
    href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
    rel="stylesheet"
    integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
    crossorigin="anonymous"
  />
  <!-- Icons -->
  <link
    rel="stylesheet"
    href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"
    referrerpolicy="no-referrer"
  />

  <style>
    :root{
      --brand:#0d6efd; --brand-2:#5aa2ff; --sidebar:#1f2430; --sidebar-hover:#2b3140; --bg:#f6f8fb;
    }
    body{ background:var(--bg); min-height:100vh; }

    /* Fixed sidebar on lg+, offcanvas on mobile */
    .sidebar-fixed {
      width: 260px;
      position: fixed;
      inset: 0 auto 0 0;
      background: var(--sidebar);
      color: #e9ecef;
      padding: 1rem .75rem;
      box-shadow: 2px 0 16px rgba(0,0,0,.12);
      display: none;
    }
    @media (min-width: 992px){
      .sidebar-fixed { display: block; }
      main { margin-left: 260px; }
    }

    .sidebar-brand {
      display:flex; align-items:center; gap:.65rem;
      padding:.75rem .5rem;
      border-bottom:1px solid rgba(255,255,255,.08);
      margin-bottom:.5rem;
    }
    .sidebar-brand img { width:38px; height:38px; object-fit:cover; border-radius:.55rem; border:2px solid rgba(255,255,255,.25); }
    .nav-sidebar .nav-link {
      color:#e9ecef; border-radius:.6rem; padding:.55rem .65rem;
      display:flex; gap:.6rem; align-items:center;
    }
    .nav-sidebar .nav-link:hover { background:var(--sidebar-hover); color:#fff; }
    .nav-sidebar .badge { background:rgba(255,255,255,.12); border:1px solid rgba(255,255,255,.2); }

    /* Appbar */
    .appbar{
      background: linear-gradient(135deg, var(--brand), var(--brand-2));
      color:#fff; padding:.85rem 1rem;
      display:flex; justify-content:space-between; align-items:center;
      box-shadow: 0 6px 24px rgba(13,110,253,.25);
      position: sticky; top: 0; z-index: 1020;
    }
    .appbar .badge{
      background: rgba(255,255,255,.15);
      color:#fff; border:1px solid rgba(255,255,255,.25);
    }

    /* KPI */
    .kpi.card { border:none; border-radius: 1rem; }
    .kpi .icon {
      width:42px; height:42px; border-radius:.8rem;
      display:flex; align-items:center; justify-content:center;
      background: rgba(13,110,253,.12); color: var(--brand);
    }
    .kpi .value { font-size:1.6rem; font-weight:800; }
    .kpi .label { color:#6c757d; }

    /* Tiles */
    .btn-tile{
      display:flex; justify-content:space-between; align-items:center; gap:.75rem;
      border-radius:.9rem; padding:.9rem 1rem; width:100%;
    }
    .btn-tile .count.badge{
      background:#fff; color:#0b1320; border:1px solid rgba(0,0,0,.1);
    }

    .section-title { font-weight:800; color:#0b1320; margin:1.25rem 0 .75rem; }
  </style>
</head>
<body>

<!-- Offcanvas sidebar (mobile) -->
<div class="offcanvas offcanvas-start" tabindex="-1" id="mobileSidebar" aria-labelledby="mobileSidebarLabel">
  <div class="offcanvas-header">
    <h6 class="offcanvas-title" id="mobileSidebarLabel">Branch Head – VVT UC</h6>
    <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
  </div>
  <div class="offcanvas-body p-0">
    <div class="px-3 pb-2">
      <a class="navbar-brand d-flex align-items-center gap-2" href="#">
        <img src="logo1.png" alt="Logo" width="32" height="32" onerror="this.style.display='none'">
        <span class="fw-semibold">Valvettithurai UC</span>
      </a>
    </div>
    <div class="nav flex-column nav-pills nav-sidebar px-2 pb-3">
      <a class="nav-link" href="view_branch_letters.php"><i class="fa-solid fa-inbox"></i><span class="ms-2">Pending Branch Letters</span><span class="badge rounded-pill ms-auto"><?= $kpi['pending']; ?></span></a>
      <a class="nav-link" href="view_received_branch.php"><i class="fa-solid fa-box-open"></i><span class="ms-2">Received (Not Assigned)</span><span class="badge rounded-pill ms-auto"><?= $kpi['received_not_assigned']; ?></span></a>
      <a class="nav-link" href="action_yes_letters.php"><i class="fa-solid fa-check-circle"></i><span class="ms-2">Action Taken</span><span class="badge rounded-pill ms-auto"><?= $kpi['action_yes']; ?></span></a>
      <a class="nav-link" href="not_seen_letters.php"><i class="fa-solid fa-eye-slash"></i><span class="ms-2">Not Seen by Staff</span><span class="badge rounded-pill ms-auto"><?= $kpi['not_seen']; ?></span></a>
      <a class="nav-link" href="seen_letters.php"><i class="fa-solid fa-magnifying-glass"></i><span class="ms-2">Seen — No Action</span><span class="badge rounded-pill ms-auto"><?= $kpi['seen_no_action']; ?></span></a>
      <a class="nav-link" href="sent_all_letters.php"><i class="fa-solid fa-paper-plane"></i><span class="ms-2">Sent — All</span><span class="badge rounded-pill ms-auto"><?= $kpi['total']; ?></span></a>
      <a class="nav-link" href="3days.php"><i class="fa-solid fa-hourglass-start"></i><span class="ms-2">3–7 Days — No Action</span><span class="badge rounded-pill ms-auto"><?= $kpi['three_to_seven_no_action']; ?></span></a>
      <hr class="my-2">
      <a class="nav-link" href="file_entry.php"><i class="fa-solid fa-file-arrow-up"></i><span class="ms-2">File Input</span></a>
      <a class="nav-link" href="file_management_view.php"><i class="fa-solid fa-folder-open"></i><span class="ms-2">Branch Files</span></a>
      <a class="nav-link" href="view_letters_statistics.php"><i class="fa-solid fa-chart-line"></i><span class="ms-2">Letters Statistics</span></a>
      <a class="nav-link" href="staff.php"><i class="fa-solid fa-users"></i><span class="ms-2">Staff</span></a>
      <a class="nav-link text-danger" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i><span class="ms-2">Logout</span></a>
    </div>
  </div>
</div>

<!-- Fixed sidebar (lg and up) -->
<aside class="sidebar-fixed d-none d-lg-block">
  <div class="sidebar-brand">
    <img src="logo1.png" alt="Logo" onerror="this.style.display='none'">
    <div>
      <div class="fw-bold">Branch Head</div>
      <small class="text-white-50">Valvettithurai UC</small>
    </div>
  </div>
  <nav class="nav flex-column nav-sidebar">
    <a class="nav-link" href="view_branch_letters.php"><i class="fa-solid fa-inbox"></i><span class="ms-2">Pending Branch Letters</span><span class="badge rounded-pill ms-auto"><?= $kpi['pending']; ?></span></a>
    <a class="nav-link" href="view_received_branch.php"><i class="fa-solid fa-box-open"></i><span class="ms-2">Received (Not Assigned)</span><span class="badge rounded-pill ms-auto"><?= $kpi['received_not_assigned']; ?></span></a>
    <a class="nav-link" href="action_yes_letters.php"><i class="fa-solid fa-check-circle"></i><span class="ms-2">Action Taken</span><span class="badge rounded-pill ms-auto"><?= $kpi['action_yes']; ?></span></a>
    <a class="nav-link" href="not_seen_letters.php"><i class="fa-solid fa-eye-slash"></i><span class="ms-2">Not Seen by Staff</span><span class="badge rounded-pill ms-auto"><?= $kpi['not_seen']; ?></span></a>
    <a class="nav-link" href="seen_letters.php"><i class="fa-solid fa-magnifying-glass"></i><span class="ms-2">Seen — No Action</span><span class="badge rounded-pill ms-auto"><?= $kpi['seen_no_action']; ?></span></a>
    <a class="nav-link" href="sent_all_letters.php"><i class="fa-solid fa-paper-plane"></i><span class="ms-2">Sent — All</span><span class="badge rounded-pill ms-auto"><?= $kpi['total']; ?></span></a>
    <a class="nav-link" href="3days.php"><i class="fa-solid fa-hourglass-start"></i><span class="ms-2">3–7 Days — No Action</span><span class="badge rounded-pill ms-auto"><?= $kpi['three_to_seven_no_action']; ?></span></a>
    <hr class="border-secondary-subtle">
    <a class="nav-link" href="file_entry.php"><i class="fa-solid fa-file-arrow-up"></i><span class="ms-2">File Input</span></a>
    <a class="nav-link" href="file_management_view.php"><i class="fa-solid fa-folder-open"></i><span class="ms-2">Branch Files</span></a>
    <a class="nav-link" href="view_letters_statistics.php"><i class="fa-solid fa-chart-line"></i><span class="ms-2">Letters Statistics</span></a>
    <a class="nav-link" href="staff.php"><i class="fa-solid fa-users"></i><span class="ms-2">Staff</span></a>
    <a class="nav-link text-danger" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i><span class="ms-2">Logout</span></a>
  </nav>
</aside>

<!-- Main -->
<main>
  <!-- App bar -->
  <div class="appbar">
    <div class="d-flex align-items-center gap-2">
      <button class="btn btn-light d-lg-none me-2" type="button"
              data-bs-toggle="offcanvas" data-bs-target="#mobileSidebar"
              aria-controls="mobileSidebar" aria-label="Open menu">
        <i class="fa-solid fa-bars"></i>
      </button>
      <div class="d-flex align-items-center gap-3">
        <img src="logo.jpg" alt="Logo" width="44" height="44"
             style="object-fit:cover;border-radius:.6rem;border:2px solid rgba(255,255,255,.35)"
             onerror="this.style.display='none'">
        <div>
          <h5 class="mb-0 fw-bold">Office Management System</h5>
          <small class="text-white-75">Valvettithurai Urban Council</small>
        </div>
      </div>
    </div>
    <div class="d-flex align-items-center flex-wrap gap-2">
      <span class="badge rounded-pill"><i class="fa-solid fa-building me-1"></i>Branch: <?= $branchName; ?></span>
      <span class="badge rounded-pill"><i class="fa-solid fa-user-tie me-1"></i><?= $username; ?></span>
      <span class="badge rounded-pill"><i class="fa-solid fa-id-badge me-1"></i>Role: Branch Head</span>

      <!-- Logout button in header -->
      <form action="logout.php" method="post" class="ms-2 mb-0"
            onsubmit="return confirm('Are you sure you want to log out?');">
        <button type="submit" class="btn btn-outline-light">
          <i class="fa-solid fa-right-from-bracket me-1"></i> Logout
        </button>
      </form>
    </div>
  </div>

  <div class="container-fluid py-3">
    <!-- KPI GRID -->
    <div class="row g-3">
      <div class="col-12 col-sm-6 col-xl-3">
        <div class="card kpi h-100 p-3">
          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="label">Total Letters</div>
              <div class="value"><?= $kpi['total']; ?></div>
            </div>
            <div class="icon"><i class="fa-solid fa-envelope-open-text"></i></div>
          </div>
        </div>
      </div>
      <div class="col-12 col-sm-6 col-xl-3">
        <div class="card kpi h-100 p-3">
          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="label">Pending (Head)</div>
              <div class="value"><?= $kpi['pending']; ?></div>
            </div>
            <div class="icon"><i class="fa-solid fa-hourglass-half"></i></div>
          </div>
        </div>
      </div>
      <div class="col-12 col-sm-6 col-xl-3">
        <div class="card kpi h-100 p-3">
          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="label">Received (Not Assigned)</div>
              <div class="value"><?= $kpi['received_not_assigned']; ?></div>
            </div>
            <div class="icon"><i class="fa-solid fa-box-open"></i></div>
          </div>
        </div>
      </div>
      <div class="col-12 col-sm-6 col-xl-3">
        <div class="card kpi h-100 p-3">
          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="label">Assigned to Staff</div>
              <div class="value"><?= $kpi['assigned']; ?></div>
            </div>
            <div class="icon"><i class="fa-solid fa-user-check"></i></div>
          </div>
        </div>
      </div>

      <div class="col-12 col-sm-6 col-xl-3">
        <div class="card kpi h-100 p-3">
          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="label">Not Seen by Staff</div>
              <div class="value"><?= $kpi['not_seen']; ?></div>
            </div>
            <div class="icon"><i class="fa-solid fa-eye-slash"></i></div>
          </div>
        </div>
      </div>
      <div class="col-12 col-sm-6 col-xl-3">
        <div class="card kpi h-100 p-3">
          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="label">Seen – No Action</div>
              <div class="value"><?= $kpi['seen_no_action']; ?></div>
            </div>
            <div class="icon"><i class="fa-solid fa-magnifying-glass"></i></div>
          </div>
        </div>
      </div>
      <div class="col-12 col-sm-6 col-xl-3">
        <div class="card kpi h-100 p-3">
          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="label">Action Taken</div>
              <div class="value"><?= $kpi['action_yes']; ?></div>
            </div>
            <div class="icon"><i class="fa-solid fa-circle-check"></i></div>
          </div>
        </div>
      </div>
      <div class="col-12 col-sm-6 col-xl-3">
        <div class="card kpi h-100 p-3">
          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="label">3–7 Days • No Action</div>
              <div class="value"><?= $kpi['three_to_seven_no_action']; ?></div>
            </div>
            <div class="icon"><i class="fa-solid fa-hourglass-start"></i></div>
          </div>
        </div>
      </div>

      <!-- Optional extra KPI: Overdue > 7 days -->
      <div class="col-12 col-sm-6 col-xl-3">
        <div class="card kpi h-100 p-3">
          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="label">Overdue &gt; 7 Days</div>
              <div class="value"><?= $kpi['overdue_no_action']; ?></div>
            </div>
            <div class="icon"><i class="fa-solid fa-triangle-exclamation"></i></div>
          </div>
        </div>
      </div>

      <!-- Optional extra KPI: Today Received -->
      <div class="col-12 col-sm-6 col-xl-3">
        <div class="card kpi h-100 p-3">
          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="label">Received Today</div>
              <div class="value"><?= $kpi['today_received']; ?></div>
            </div>
            <div class="icon"><i class="fa-solid fa-calendar-day"></i></div>
          </div>
        </div>
      </div>
    </div>

    <!-- LETTER MANAGEMENT -->
    <h6 class="section-title">Letter Management</h6>
    <div class="row g-3">
      <div class="col-12 col-md-6 col-xl-3">
        <a href="view_branch_letters.php" class="btn btn-primary btn-tile">
          <span class="text-start">
            <div class="fw-bold">Pending Letters</div>
            <small>Awaiting Branch Head decision</small>
          </span>
          <span class="badge rounded-pill count"><?= $kpi['pending']; ?></span>
        </a>
      </div>

      <div class="col-12 col-md-6 col-xl-3">
        <a href="view_received_branch.php" class="btn btn-danger btn-tile">
          <span class="text-start">
            <div class="fw-bold">Received (Not Assigned)</div>
            <small>Needs assignment to staff</small>
          </span>
          <span class="badge rounded-pill count"><?= $kpi['received_not_assigned']; ?></span>
        </a>
      </div>

      <div class="col-12 col-md-6 col-xl-3">
        <a href="not_seen_letters.php" class="btn btn-dark btn-tile">
          <span class="text-start">
            <div class="fw-bold">Not Seen by Staff</div>
            <small>Assigned but unseen</small>
          </span>
          <span class="badge rounded-pill count"><?= $kpi['not_seen']; ?></span>
        </a>
      </div>

      <div class="col-12 col-md-6 col-xl-3">
        <a href="seen_letters.php" class="btn btn-warning btn-tile">
          <span class="text-start">
            <div class="fw-bold">Seen – No Action</div>
            <small>Follow-up required</small>
          </span>
          <span class="badge rounded-pill count"><?= $kpi['seen_no_action']; ?></span>
        </a>
      </div>

      <div class="col-12 col-md-6 col-xl-3">
        <a href="action_yes_letters.php" class="btn btn-info btn-tile">
          <span class="text-start">
            <div class="fw-bold">Action Taken Letters</div>
            <small>Completed by staff</small>
          </span>
          <span class="badge rounded-pill count"><?= $kpi['action_yes']; ?></span>
        </a>
      </div>

      <div class="col-12 col-md-6 col-xl-3">
        <a href="sent_all_letters.php" class="btn btn-secondary btn-tile">
          <span class="text-start">
            <div class="fw-bold">Sent — All</div>
            <small>All outgoing from branch</small>
          </span>
          <span class="badge rounded-pill count"><?= $kpi['total']; ?></span>
        </a>
      </div>

      <div class="col-12 col-md-6 col-xl-3">
        <a href="3days.php" class="btn btn-outline-secondary btn-tile">
          <span class="text-start">
            <div class="fw-bold">3–7 Days • No Action</div>
            <small>Time-sensitive follow-up</small>
          </span>
          <span class="badge rounded-pill count"><?= $kpi['three_to_seven_no_action']; ?></span>
        </a>
      </div>
    </div>

    <!-- FILE MANAGEMENT -->
    <h6 class="section-title">File Management</h6>
    <div class="row g-3">
      <div class="col-12 col-md-6 col-xl-3">
        <a href="file_management_view.php" class="btn btn-primary btn-tile">
          <span class="text-start">
            <div class="fw-bold">View Branch Files</div>
            <small>Browse uploaded files</small>
          </span>
          <i class="fa-solid fa-folder-open"></i>
        </a>
      </div>

      <div class="col-12 col-md-6 col-xl-3">
        <a href="file_entry.php" class="btn btn-success btn-tile">
          <span class="text-start">
            <div class="fw-bold">File Input</div>
            <small>Upload new file</small>
          </span>
          <i class="fa-solid fa-file-arrow-up"></i>
        </a>
      </div>

      <div class="col-12 col-md-6 col-xl-3">
        <a href="out_letter/view_drafted_letters.php" class="btn btn-secondary btn-tile">
          <span class="text-start">
            <div class="fw-bold">Drafted Letters</div>
            <small>Outbound drafts</small>
          </span>
          <i class="fa-solid fa-file-pen"></i>
        </a>
      </div>

      <div class="col-12 col-md-6 col-xl-3">
        <a href="out_letter/view_approved_letters.php" class="btn btn-info btn-tile">
          <span class="text-start">
            <div class="fw-bold">Approved Letters</div>
            <small>Ready to send</small>
          </span>
          <i class="fa-solid fa-circle-check"></i>
        </a>
      </div>

      <div class="col-12 col-md-6 col-xl-3">
        <a href="out_letter/view_rejected_letters.php" class="btn btn-danger btn-tile">
          <span class="text-start">
            <div class="fw-bold">Rejected Letters</div>
            <small>Needs edits</small>
          </span>
          <i class="fa-solid fa-circle-xmark"></i>
        </a>
      </div>
    </div>
  </div>
</main>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
        integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
        crossorigin="anonymous"></script>
</body>
</html>
