<?php
// Start the session
session_start();

// Include the database connection
include 'db_connection.php'; // Ensure this is the correct path to your db_connection.php

// Initialize error and success messages
$error_message = "";
$success_message = "";

// Check if the form is submitted
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Retrieve data from the form
    $branch_name = $_POST['branch_name'];

    // Validate form data
    if (empty($branch_name)) {
        $error_message = "Please fill in the branch name!";
    } else {
        // Prepare the SQL query to insert the data (branch_id will auto-increment)
        $query = "INSERT INTO branches (branch_name) VALUES (?)";
        
        if ($stmt = $conn->prepare($query)) {
            // Bind the parameter
            $stmt->bind_param("s", $branch_name);

            // Execute the query
            if ($stmt->execute()) {
                $success_message = "Branch data added successfully!";
            } else {
                $error_message = "Error: " . $stmt->error;
            }

            // Close the statement
            $stmt->close();
        } else {
            $error_message = "Error preparing query: " . $conn->error;
        }
    }

    // Close the connection
    $conn->close();
}
?>
<!doctype html>
<html lang="en" data-bs-theme="light">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Branch Data Entry · Office Management System</title>

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <!-- Icons (optional) -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">

  <style>
    body{
      min-height:100vh;
      background:#f4f7fb;
      display:flex; align-items:center; justify-content:center;
      padding: 24px 12px;
    }
    .card{
      width:100%;
      max-width: 560px;
      border:0; border-radius:16px;
      box-shadow: 0 12px 28px rgba(0,0,0,.12);
      overflow:hidden;
    }
    .card-header{
      background:#fff; border-bottom:1px solid rgba(0,0,0,.06);
    }
  </style>
</head>
<body>

<div class="card">
  <div class="card-header">
    <div class="d-flex align-items-center justify-content-between">
      <h1 class="h5 m-0">Enter Branch Information</h1>
      <a href="super_admin_dashboard.php" class="btn btn-sm btn-outline-secondary">
        <i class="bi bi-arrow-left"></i> Back
      </a>
    </div>
  </div>

  <div class="card-body">
    <!-- Success / Error alerts -->
    <?php if (!empty($success_message)): ?>
      <div class="alert alert-success d-flex align-items-center" role="alert">
        <i class="bi bi-check-circle me-2"></i>
        <div><?= htmlspecialchars($success_message, ENT_QUOTES, 'UTF-8'); ?></div>
      </div>
    <?php endif; ?>

    <?php if (!empty($error_message)): ?>
      <div class="alert alert-danger d-flex align-items-center" role="alert">
        <i class="bi bi-exclamation-triangle me-2"></i>
        <div><?= htmlspecialchars($error_message, ENT_QUOTES, 'UTF-8'); ?></div>
      </div>
    <?php endif; ?>

    <form action="" method="POST" class="mt-2" id="branchForm" novalidate>
      <div class="mb-3">
        <label for="branch_name" class="form-label">Branch Name <span class="text-danger">*</span></label>
        <input
          type="text"
          class="form-control"
          name="branch_name"
          id="branch_name"
          required
          minlength="2"
          placeholder="e.g., Accounts, Engineering, Public Works"
          value="<?= isset($_POST['branch_name']) ? htmlspecialchars($_POST['branch_name'], ENT_QUOTES, 'UTF-8') : '' ?>"
        >
        <div class="form-text">Minimum 2 characters. Use official branch naming.</div>
      </div>

      <div class="d-flex gap-2">
        <button type="submit" class="btn btn-primary">
          <i class="bi bi-save"></i> Submit
        </button>
        <button type="reset" class="btn btn-outline-secondary">
          Reset
        </button>
      </div>
    </form>
  </div>
</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Autofocus + trim + prevent double submit (frontend only; backend unchanged)
  const form = document.getElementById('branchForm');
  const input = document.getElementById('branch_name');
  input?.focus();

  form?.addEventListener('submit', function(e){
    // quick trim
    if (input) input.value = input.value.trim();

    // basic HTML5 validation gate
    if (!form.checkValidity()){
      e.preventDefault();
      e.stopPropagation();
      form.classList.add('was-validated');
      return;
    }

    // prevent double submit
    const btn = form.querySelector('button[type="submit"]');
    if (btn){
      btn.disabled = true;
      btn.innerHTML = '<span class="spinner-border spinner-border-sm me-1" role="status" aria-hidden="true"></span> Saving...';
    }
  });
</script>
</body>
</html>
