<?php
session_start();

// Require role
if (!isset($_SESSION['role'])) {
    header('Location: unauthorized.php');
    exit();
}

require_once 'db_connection.php'; // provides $conn (mysqli), utf8mb4

$role          = $_SESSION['role'];
$user_branchId = isset($_SESSION['branch_id']) ? (int)$_SESSION['branch_id'] : 0;

// --- Read filters safely ---
$start_date = isset($_POST['start_date']) ? trim($_POST['start_date']) : '';
$end_date   = isset($_POST['end_date'])   ? trim($_POST['end_date'])   : '';
$branch_id  = isset($_POST['branch_id'])  ? (int)$_POST['branch_id']   : 0;
$priority   = isset($_POST['priority'])   ? trim($_POST['priority'])   : '';

// Base conditions: unseen letters
$conditions = ["letters.seen = 0"];
$params     = [];
$types      = "";

// Role scoping
$adminLike = ($role === 'letter_admin' || $role === 'super_admin');

if ($role === 'branch_head') {
    // Branch heads only see their branch
    $conditions[] = "letters.branch_id = ?";
    $params[] = $user_branchId;
    $types .= "i";
}

// Admin-like users can optionally filter by branch
if ($adminLike && $branch_id > 0) {
    $conditions[] = "letters.branch_id = ?";
    $params[] = $branch_id;
    $types .= "i";
}

// Date range (validate YYYY-MM-DD)
if ($start_date !== '' && $end_date !== '') {
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $start_date) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $end_date)) {
        $conditions[] = "letters.letter_date BETWEEN ? AND ?";
        $params[] = $start_date; $params[] = $end_date;
        $types .= "ss";
    }
}

// Priority filter
if ($priority !== '') {
    $conditions[] = "letters.priority = ?";
    $params[] = $priority;
    $types .= "s";
}

$where = "WHERE " . implode(" AND ", $conditions);

// Main query
$sql = "
SELECT
  letters.letter_id,
  letters.letter_number,
  letters.letter_date,
  letters.department,
  letters.subject,
  branches.branch_name,
  letters.scan_copy,
  letters.forward_to,
  u.username AS staff_name,
  letters.seen_time,
  letters.seen,
  letters.action,
  letters.hod_comment,
  letters.priority
FROM letters
JOIN branches ON letters.branch_id = branches.branch_id
LEFT JOIN users u ON letters.forward_to = u.user_id
{$where}
ORDER BY letters.letter_date DESC, letters.letter_id DESC
";

$stmt = $conn->prepare($sql);
if ($types !== "") { $stmt->bind_param($types, ...$params); }
$stmt->execute();
$result = $stmt->get_result();

// Branch list for admin-like users’ filter
$branches = null;
if ($adminLike) {
    $branches = $conn->query("SELECT branch_id, branch_name FROM branches ORDER BY branch_name ASC");
}
$priorities = ['High', 'Medium', 'Low'];

// keep connection open if needed
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Unseen Letters</title>

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
        rel="stylesheet"
        integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
        crossorigin="anonymous"/>
  <link rel="stylesheet"
        href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"/>

  <style>
    :root { --brand:#0d6efd; --bg:#f6f8fb; }
    body { background: var(--bg); padding-bottom: 40px; }
    header.page-head{
      background: linear-gradient(135deg, var(--brand), #5aa2ff);
      color: #fff; border-radius: 1rem; padding: 1rem 1.25rem;
      box-shadow: 0 6px 24px rgba(13,110,253,.25);
      margin-bottom: 1rem;
    }
    .card-filter { border: none; border-radius: 1rem; }
    .table thead th { white-space: nowrap; }
    .high-priority { background-color: #ffe1e1; } /* softer red */
    .btn-reset { white-space: nowrap; }
  </style>
</head>
<body>

<header class="page-head">
  <div class="d-flex align-items-center justify-content-between">
    <div class="d-flex align-items-center gap-3">
      <i class="fa-solid fa-eye-slash fs-3"></i>
      <div>
        <h5 class="mb-0 fw-bold">Unseen Letters by Staff</h5>
        <small>Letters where staff have <strong>not</strong> opened them yet</small>
      </div>
    </div>
    <div class="d-flex gap-2">
      <?php if ($role === 'letter_admin'): ?>
        <a class="btn btn-light" href="letter_admin_dashboard.php">
          <i class="fa-solid fa-arrow-left-long me-1"></i> Dashboard
        </a>
      <?php elseif ($role === 'branch_head'): ?>
        <a class="btn btn-light" href="branch_head_dashboard.php">
          <i class="fa-solid fa-arrow-left-long me-1"></i> Dashboard
        </a>
      <?php elseif ($role === 'super_admin'): ?>
        <a class="btn btn-light" href="super_admin_dashboard.php">
          <i class="fa-solid fa-arrow-left-long me-1"></i> Dashboard
        </a>
      <?php else: ?>
        <a class="btn btn-light" href="staff_user_dashboard.php">
          <i class="fa-solid fa-arrow-left-long me-1"></i> Dashboard
        </a>
      <?php endif; ?>
    </div>
  </div>
</header>

<div class="container-fluid">
  <!-- Filters -->
  <div class="card card-filter shadow-sm mb-3">
    <div class="card-body">
      <form action="not_seen_letters.php" method="POST" class="row g-3 align-items-end">

        <div class="col-12 col-md-3">
          <label class="form-label" for="start_date">Start Date</label>
          <input type="date" id="start_date" name="start_date" class="form-control"
                 value="<?= htmlspecialchars($start_date, ENT_QUOTES, 'UTF-8'); ?>">
        </div>

        <div class="col-12 col-md-3">
          <label class="form-label" for="end_date">End Date</label>
          <input type="date" id="end_date" name="end_date" class="form-control"
                 value="<?= htmlspecialchars($end_date, ENT_QUOTES, 'UTF-8'); ?>">
        </div>

        <?php if ($adminLike): ?>
          <div class="col-12 col-md-3">
            <label class="form-label" for="branch_id">Branch</label>
            <select id="branch_id" name="branch_id" class="form-select">
              <option value="">All Branches</option>
              <?php if ($branches && $branches->num_rows > 0): ?>
                <?php while ($b = $branches->fetch_assoc()): ?>
                  <option value="<?= (int)$b['branch_id']; ?>"
                          <?= ($branch_id === (int)$b['branch_id']) ? 'selected' : '' ?>>
                    <?= htmlspecialchars($b['branch_name'], ENT_QUOTES, 'UTF-8'); ?>
                  </option>
                <?php endwhile; ?>
              <?php endif; ?>
            </select>
          </div>
        <?php endif; ?>

        <div class="col-12 col-md-3">
          <label class="form-label" for="priority">Priority</label>
          <select id="priority" name="priority" class="form-select">
            <option value="">All Priorities</option>
            <?php foreach (['High','Medium','Low'] as $p): ?>
              <option value="<?= htmlspecialchars($p, ENT_QUOTES, 'UTF-8'); ?>"
                      <?= ($priority === $p ? 'selected' : '') ?>>
                <?= htmlspecialchars($p, ENT_QUOTES, 'UTF-8'); ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-auto">
          <button type="submit" class="btn btn-primary">
            <i class="fa-solid fa-filter me-1"></i> Apply Filters
          </button>
        </div>
        <div class="col-auto">
          <a href="not_seen_letters.php" class="btn btn-outline-secondary btn-reset">
            <i class="fa-solid fa-rotate-left me-1"></i> Reset
          </a>
        </div>
      </form>
    </div>
  </div>

  <!-- Quick search -->
  <div class="row mb-2">
    <div class="col-12 col-md-6 ms-auto">
      <input type="text" id="searchInput" class="form-control" placeholder="Search in table…" onkeyup="searchTable()">
    </div>
  </div>

  <!-- Table -->
  <div class="card shadow-sm">
    <div class="table-responsive">
      <table class="table table-striped align-middle mb-0" id="lettersTable">
        <thead class="table-light">
          <tr>
            <th>Letter ID</th>
            <th>Letter No.</th>
            <th>Letter Date</th>
            <th>Department</th>
            <th>Subject</th>
            <th>Branch</th>
            <th>Staff (Forward To)</th>
            <th>Scan</th>
            <th>Seen</th>
            <th>Action</th>
            <th>HOD Comment</th>
            <th>Priority</th>
          </tr>
        </thead>
        <tbody>
        <?php if ($result && $result->num_rows > 0): ?>
          <?php while ($row = $result->fetch_assoc()): ?>
            <?php $isHigh = (isset($row['priority']) && $row['priority'] === 'High'); ?>
            <tr class="<?= $isHigh ? 'high-priority' : '' ?>">
              <td><?= (int)$row['letter_id']; ?></td>
              <td><?= htmlspecialchars($row['letter_number'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['letter_date'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['department'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['subject'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['branch_name'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['staff_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
              <td>
                <?php if (!empty($row['scan_copy'])): ?>
                  <a class="btn btn-sm btn-outline-primary"
                     href="<?= htmlspecialchars($row['scan_copy'], ENT_QUOTES, 'UTF-8'); ?>"
                     target="_blank"><i class="fa-solid fa-up-right-from-square me-1"></i>View</a>
                <?php else: ?>
                  <span class="text-muted">N/A</span>
                <?php endif; ?>
              </td>
              <td><?= ($row['seen'] ?? 0) == 1 ? 'Yes' : 'No'; ?></td>
              <td><?= htmlspecialchars($row['action'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['hod_comment'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['priority'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
            </tr>
          <?php endwhile; ?>
        <?php else: ?>
          <tr><td colspan="12" class="text-center py-4">No unseen letters found.</td></tr>
        <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>

  <!-- Back Button -->
  <div class="mt-3">
    <?php if ($role === 'letter_admin'): ?>
      <a href="letter_admin_dashboard.php" class="btn btn-light">
        <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
      </a>
    <?php elseif ($role === 'branch_head'): ?>
      <a href="branch_head_dashboard.php" class="btn btn-light">
        <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
      </a>
    <?php elseif ($role === 'super_admin'): ?>
      <a href="super_admin_dashboard.php" class="btn btn-light">
        <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
      </a>
    <?php else: ?>
      <a href="staff_user_dashboard.php" class="btn btn-light">
        <i class="fa-solid fa-arrow-left-long me-1"></i> Back to Dashboard
      </a>
    <?php endif; ?>
  </div>
</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
        integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
        crossorigin="anonymous"></script>

<!-- Simple client-side search -->
<script>
function searchTable() {
  const input = document.getElementById("searchInput");
  const filter = (input.value || "").toUpperCase();
  const table = document.getElementById("lettersTable");
  const tr = table.getElementsByTagName("tr");

  for (let i = 1; i < tr.length; i++) {
    tr[i].style.display = "none";
    const tds = tr[i].getElementsByTagName("td");
    for (let j = 0; j < tds.length; j++) {
      const txt = (tds[j].textContent || tds[j].innerText || "").toUpperCase();
      if (txt.indexOf(filter) > -1) { tr[i].style.display = ""; break; }
    }
  }
}
</script>
</body>
</html>
