<?php
// Start the session at the beginning of the file
session_start();

// Check if the user is logged in and $_SESSION['user_id'] is set
if (!isset($_SESSION['user_id'])) {
    echo "<p>Error: You need to be logged in to draft a letter.</p>";
    exit;
}

// Include database connection
include '..\db_connection.php';

// Fetch the branch ID from session
$branch_id = $_SESSION['branch_id'];

// Fetch the branch name based on session's branch_id to auto-fill the branch field
$branch_query = "SELECT branch_name FROM branches WHERE branch_id = ?";
$stmt = $conn->prepare($branch_query);
$stmt->bind_param("i", $branch_id);
$stmt->execute();
$stmt->bind_result($branch_name);
$stmt->fetch();
$stmt->close();

// Fetch all users who are approvers (e.g., branch heads, super admins) based on the branch_id
$query = "SELECT u.user_id, u.username, b.branch_name
          FROM users u 
          JOIN branches b ON u.branch_id = b.branch_id 
          WHERE u.role IN ('branch_head', 'super_admin') AND u.branch_id = ?";
$approvers_result = $conn->prepare($query);
$approvers_result->bind_param("i", $branch_id);
$approvers_result->execute();
$approvers = $approvers_result->get_result();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Get form data
    $letter_number = $_POST['letter_number'];
    $letter_date = $_POST['letter_date'];
    $subject = $_POST['subject'];
    $department = $_POST['department'];
    $staff_user_id = $_SESSION['user_id'];  // Assuming the staff user is logged in
    $approver_id = $_POST['approver_id'];  // Approver selected by the staff user

    // Handle file upload
    $attachment = null;
    if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] == 0) {
        $file_name = $_FILES['attachment']['name'];
        $file_tmp = $_FILES['attachment']['tmp_name'];
        $file_size = $_FILES['attachment']['size'];
        $file_ext = pathinfo($file_name, PATHINFO_EXTENSION);

        // Define allowed file extensions
        $allowed_extensions = ['pdf', 'docx', 'doc', 'jpg', 'png'];

        // Check if the file type is allowed
        if (in_array($file_ext, $allowed_extensions)) {
            // Define a directory to store the uploaded file
            $upload_dir = 'upload/';
            $file_path = $upload_dir . basename($file_name);

            // Check if the directory exists, create it if not
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }

            // Move the uploaded file to the server directory
            if (move_uploaded_file($file_tmp, $file_path)) {
                $attachment = $file_path;
            } else {
                echo "<p>Error: Unable to upload the file.</p>";
            }
        } else {
            echo "<p>Error: Invalid file type. Only PDF, DOCX, DOC, JPG, PNG files are allowed.</p>";
        }
    }

    // Insert the data into `letter_drafts` table
    $stmt = $conn->prepare("INSERT INTO letter_drafts (letter_number, letter_date, subject, department, staff_user_id, approver_id, attachment, branch_id) 
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("ssssssss", $letter_number, $letter_date, $subject, $department, $staff_user_id, $approver_id, $attachment, $branch_id);

    if ($stmt->execute()) {
        echo "<p>Draft letter saved successfully!</p>";
    } else {
        echo "<p>Error: " . $stmt->error . "</p>";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Draft Letter</title>
    <link rel="stylesheet" href="https://maxcdn.bootstrapcdn.com/bootstrap/4.0.0/css/bootstrap.min.css">
    <style>
        body {
            background-color: #f4f7fc;
            font-family: Arial, sans-serif;
        }
        .container {
            max-width: 800px;
            margin-top: 50px;
        }
        h2 {
            color: #333;
            font-weight: bold;
        }
        .form-group label {
            font-weight: bold;
            color: #444;
        }
        .form-control {
            border-radius: 5px;
            border: 1px solid #ccc;
            padding: 12px;
        }
        .btn-primary {
            background-color: #007bff;
            border-color: #007bff;
            font-size: 16px;
            padding: 10px 20px;
            width: 100%;
        }
        .form-group {
            margin-bottom: 20px;
        }
        .section-header {
            margin-top: 20px;
            font-size: 22px;
            font-weight: bold;
            color: #5c5c5c;
            text-align: center;
        }
        .alert {
            margin-top: 20px;
            padding: 10px;
            background-color: #28a745;
            color: white;
            text-align: center;
        }
    </style>
</head>
<body>
            

    <div class="container">
        <!-- Back Button -->
        <div class="mt-4">
            <a href="../staff_user_dashboard.php" class="btn btn-secondary">Back</a> <!-- Adjust the link here -->
        </div>

        <h2>Create Draft Letter</h2>
        <form action="draft_letter_form.php" method="POST" class="mt-4" enctype="multipart/form-data">
            
            <div class="form-group">
                <label for="letter_number">Letter Number</label>
                <input type="text" name="letter_number" id="letter_number" class="form-control" required>
            </div>

            <div class="form-group">
                <label for="letter_date">Letter Date</label>
                <input type="date" name="letter_date" id="letter_date" class="form-control" required>
            </div>

            <div class="form-group">
                <label for="subject">Subject</label>
                <input type="text" name="subject" id="subject" class="form-control" required>
            </div>

            <div class="form-group">
                <label for="department">Department</label>
                <input type="text" name="department" id="department" class="form-control" required>
            </div>

            <div class="form-group">
                <label for="branch_id">Branch</label>
                <input type="text" name="branch_id" id="branch_id" class="form-control" value="<?php echo $branch_name; ?>" readonly>
            </div>

            <div class="form-group">
                <label for="approver_id">Approver</label>
                <select name="approver_id" id="approver_id" class="form-control" required>
                    <option value="">Select Approver</option>
                    <?php while ($row = $approvers->fetch_assoc()): ?>
                        <option value="<?php echo $row['user_id']; ?>"><?php echo $row['username'] . ' (' . $row['branch_name'] . ')'; ?></option>
                    <?php endwhile; ?>
                </select>
            </div>

            <div class="form-group">
                <label for="attachment">Attachment</label>
                <input type="file" name="attachment" id="attachment" class="form-control" accept=".pdf, .docx, .doc, .jpg, .png">
            </div>

            <button type="submit" class="btn btn-primary">Save Draft</button>
        </form>
    </div>

    <script src="https://code.jquery.com/jquery-3.2.1.slim.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/popper.js/1.12.9/umd/popper.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.0.0/js/bootstrap.min.js"></script>

</body>
</html>
