<?php
// Include the database connection file
include('db_connection.php');

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Check if the 'selected_letters' key exists in the POST data
    if (isset($_POST['selected_letters']) && !empty($_POST['selected_letters'])) {
        // Get selected letter IDs and the user to forward to
        $selectedLetters = $_POST['selected_letters'];  // Array of selected letter IDs
        $forwardToUser = $_POST['forward_to_user'];  // User ID to forward to
        
        // Ensure that a user is selected to forward the letters to
        if (isset($forwardToUser)) {
            // Prepare the SQL query to update the forward_to field
            $placeholders = implode(',', array_fill(0, count($selectedLetters), '?'));
            $query = "UPDATE letters SET forward_to = ? WHERE letter_id IN ($placeholders)";
            
            // Prepare and bind parameters
            $stmt = $conn->prepare($query);
            
            // First parameter is the user ID, followed by the selected letter IDs
            $params = array_merge([$forwardToUser], $selectedLetters);
            $stmt->bind_param(str_repeat('i', count($params)), ...$params);
            
            // Execute the query
            if ($stmt->execute()) {
                echo "<p style='color:green;'>Letters successfully forwarded.</p>";
            } else {
                echo "<p style='color:red;'>Error forwarding letters: " . $stmt->error . "</p>";
            }
            
            $stmt->close();
        } else {
            echo "<p style='color:red;'>Please select a user to forward to.</p>";
        }
    } else {
        echo "<p style='color:red;'>Please select at least one letter to forward.</p>";
    }
}

// Fetching letters and users for the form
$lettersQuery = "SELECT l.*, b.branch_name, u.username AS branch_head, f.username AS forward_to_user FROM letters l
                 LEFT JOIN branches b ON l.branch_id = b.branch_id
                 LEFT JOIN users u ON l.user_id = u.user_id
                 LEFT JOIN users f ON l.forward_to = f.user_id";
$lettersResult = $conn->query($lettersQuery);

$usersQuery = "SELECT user_id, username FROM users WHERE role IN ('branch_head', 'staff_user')";
$usersResult = $conn->query($usersQuery);
?>

<!DOCTYPE html>
<html lang="en" data-bs-theme="light">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Forward Letters</title>
    
    <!-- Bootstrap 5 -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">

    <!-- DataTables (Bootstrap 5 theme) -->
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/responsive/2.5.0/css/responsive.bootstrap5.min.css">

    <!-- Select2 -->
    <link href="https://cdn.jsdelivr.net/npm/select2@4.0.13/dist/css/select2.min.css" rel="stylesheet" />

    <!-- Bootstrap Icons (optional for small icons) -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">

    <style>
        body { background:#f4f7fb; }
        .page-wrap{ max-width:1200px; margin:0 auto; padding:24px 12px 48px; }
        .card{ border:0; border-radius:16px; box-shadow:0 10px 24px rgba(0,0,0,.08); }
        .select2-container .select2-selection--single{
          height: 38px; padding: 4px 8px;
          border: 1px solid #ced4da; border-radius:.375rem;
        }
        .select-all-wrap{ display:flex; align-items:center; gap:.5rem; }
        .table thead th { font-weight:600; }
        .badge-priority{ font-weight:600; }
        .top-actions{ gap:.5rem; }
    </style>
</head>
<body>

<div class="page-wrap">
    <div class="d-flex align-items-center justify-content-between flex-wrap top-actions mb-3">
        <a href="super_admin_dashboard.php" class="btn btn-outline-secondary"><i class="bi bi-arrow-left"></i> Back to dashboard</a>
        <h1 class="h4 m-0">Forward Letters</h1>
        <div></div>
    </div>

    <div class="card">
        <div class="card-body">
            <h2 class="h5 mb-3">Select Letters to Forward</h2>

            <form method="POST" action="" id="forwardForm">
                <!-- Controls row -->
                <div class="row g-3 align-items-end mb-2">
                    <div class="col-sm-8">
                        <label for="forward_to_user" class="form-label">Select User to Forward To</label>
                        <select name="forward_to_user" id="forward_to_user" class="form-select">
                            <option value="">Select User</option>
                            <?php
                            if ($usersResult->num_rows > 0) {
                                while ($user = $usersResult->fetch_assoc()) {
                                    echo "<option value='{$user['user_id']}'>{$user['username']}</option>";
                                }
                            } else {
                                echo "<option value=''>No users available</option>";
                            }
                            ?>
                        </select>
                    </div>
                    <div class="col-sm-4">
                        <label class="form-label d-none d-sm-block">&nbsp;</label>
                        <button type="button" id="confirmForwardBtn" class="btn btn-primary w-100">
                            <i class="bi bi-send"></i> Forward Letters
                        </button>
                    </div>
                </div>

                <!-- Select All -->
                <div class="select-all-wrap mb-2">
                    <input type="checkbox" id="selectAll" class="form-check-input">
                    <label for="selectAll" class="form-check-label m-0">Select All</label>
                </div>

                <!-- Table -->
                <div class="table-responsive">
                    <table id="lettersTable" class="table table-striped table-hover align-middle nowrap" style="width:100%">
                        <thead class="table-light">
                            <tr>
                                <th>Select</th>
                                <th>Letter Number</th>
                                <th>Department</th>
                                <th>Subject</th>
                                <th>Priority</th>
                                <th>Branch</th>
                                <th>Branch Head</th>
                                <th>Forward To</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            if ($lettersResult->num_rows > 0) {
                                while ($row = $lettersResult->fetch_assoc()) {
                                    // Just visual badges (no logic change): show priority with a color tone
                                    $priorityBadge = $row['priority'];
                                    $priClass = 'secondary';
                                    if (is_string($row['priority'])) {
                                        $p = strtolower($row['priority']);
                                        if ($p === 'high')   $priClass = 'danger';
                                        if ($p === 'medium') $priClass = 'warning';
                                        if ($p === 'low')    $priClass = 'success';
                                    }

                                    echo "<tr>";
                                    echo "<td><input class='form-check-input' type='checkbox' name='selected_letters[]' value='{$row['letter_id']}' /></td>";
                                    echo "<td>{$row['letter_number']}</td>";
                                    echo "<td>{$row['department']}</td>";
                                    echo "<td>{$row['subject']}</td>";
                                    echo "<td><span class='badge text-bg-{$priClass} badge-priority'>{$priorityBadge}</span></td>";
                                    echo "<td>{$row['branch_name']}</td>";
                                    echo "<td>{$row['branch_head']}</td>";
                                    echo "<td>" . ($row['forward_to_user'] ? $row['forward_to_user'] : '<span class=\"text-muted\">Not Forwarded</span>') . "</td>";
                                    echo "</tr>";
                                }
                            } else {
                                echo "<tr><td colspan='8' class='text-center text-muted py-4'>No letters found.</td></tr>";
                            }
                            ?>
                        </tbody>
                    </table>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Scripts: jQuery, Bootstrap, DataTables, Select2 -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>

<script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/dataTables.responsive.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/responsive.bootstrap5.min.js"></script>

<script src="https://cdn.jsdelivr.net/npm/select2@4.0.13/dist/js/select2.min.js"></script>

<script>
    // Initialize the DataTable
    $(document).ready(function () {
        const dt = $('#lettersTable').DataTable({
            responsive: true,
            pageLength: 10,
            order: [[1, 'desc']], // sort by Letter Number by default
            columnDefs: [{ orderable: false, targets: 0 }]
        });

        // Select All checkbox functionality
        $('#selectAll').on('click', function () {
            var isChecked = $(this).prop('checked');
            $('#lettersTable input[type="checkbox"]').prop('checked', isChecked);
        });

        // Initialize Select2 for searchable dropdown
        $('#forward_to_user').select2({
            placeholder: "Search for a user",
            allowClear: true,
            width: '100%'
        });

        // Forward button confirmation
        $('#confirmForwardBtn').on('click', function () {
            var selectedLetters = $('input[name="selected_letters[]"]:checked').length;
            if ($('#forward_to_user').val() === "" || $('#forward_to_user').val() === null) {
                alert('Please select a user to forward to.');
                return;
            }
            if (selectedLetters > 0) {
                var confirmation = confirm('Are you sure you want to forward the selected letters?');
                if (confirmation) {
                    $('#forwardForm').submit();
                }
            } else {
                alert('Please select at least one letter to forward.');
            }
        });
    });
</script>

</body>
</html>
