<?php
session_start();

if (!isset($_SESSION['user_id']) || !isset($_SESSION['role'])) {
    header("Location: login.php");
    exit();
}

require_once 'db_connection.php'; // provides $conn (mysqli), utf8mb4

$role          = $_SESSION['role'];
$user_id       = (int)($_SESSION['user_id'] ?? 0);
$user_branchId = (int)($_SESSION['branch_id'] ?? 0);

/* -------- Build secure, role-scoped query -------- */
$conditions = [];
$params = [];
$types  = "";

if ($role === 'branch_head') {
    $conditions[] = "letters.branch_id = ?";
    $params[] = $user_branchId;
    $types .= "i";
} elseif ($role === 'staff_user') {
    // align with rest of app: assignee is 'forward_to'
    $conditions[] = "letters.forward_to = ?";
    $params[] = $user_id;
    $types .= "i";
}
// letter_admin / super_admin: no extra condition (see all)

$where = $conditions ? ("WHERE " . implode(" AND ", $conditions)) : "";

$sql = "
SELECT
  letters.letter_id,
  letters.letter_number,
  letters.letter_date,
  letters.department,
  letters.subject,
  letters.received_date,
  letters.scan_copy,
  letters.action,
  letters.description,
  letters.branch_id,
  branches.branch_name,
  letters.forward_to
FROM letters
LEFT JOIN branches ON letters.branch_id = branches.branch_id
{$where}
ORDER BY letters.letter_date DESC, letters.letter_id DESC
";

$stmt = $conn->prepare($sql);
if ($types !== "") { $stmt->bind_param($types, ...$params); }
$stmt->execute();
$result = $stmt->get_result();
$stmt->close();
// $conn->close(); // keep open if you extend the page
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>View All Letters</title>

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
        rel="stylesheet"
        integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
        crossorigin="anonymous"/>
  <!-- DataTables (Bootstrap 5 theme) -->
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css"/>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"/>

  <style>
    :root { --brand:#0d6efd; --bg:#f6f8fb; }
    body { background: var(--bg); padding-bottom: 40px; }
    .page-head{
      background: linear-gradient(135deg, var(--brand), #5aa2ff);
      color: #fff; border-radius: 1rem; padding: 1rem 1.25rem;
      box-shadow: 0 6px 24px rgba(13,110,253,.25);
      margin: 1rem auto;
    }
    .table thead th { white-space: nowrap; }
  </style>
</head>
<body>
<div class="container-fluid">
  <!-- Header -->
  <div class="page-head d-flex align-items-center justify-content-between">
    <div class="d-flex align-items-center gap-3">
      <i class="fa-solid fa-envelope-open-text fs-3"></i>
      <div>
        <h5 class="mb-0 fw-bold">All Letters</h5>
        <small class="text-white-75">
          <?php if ($role === 'letter_admin' || $role === 'super_admin'): ?>
            Viewing: <strong>All Branches</strong>
          <?php elseif ($role === 'branch_head'): ?>
            Branch ID: <span class="badge bg-light text-dark"><?= htmlspecialchars((string)$user_branchId, ENT_QUOTES, 'UTF-8') ?></span>
          <?php else: ?>
            Assigned to you
          <?php endif; ?>
        </small>
      </div>
    </div>
    <div class="d-flex gap-2">
      <?php
        if ($role === 'letter_admin')       $back = 'letter_admin_dashboard.php';
        elseif ($role === 'branch_head')    $back = 'branch_head_dashboard.php';
        elseif ($role === 'staff_user')     $back = 'staff_user_dashboard.php';
        elseif ($role === 'super_admin')    $back = 'super_admin_dashboard.php';
        else                                $back = 'index.php';
      ?>
      <a class="btn btn-light" href="<?= $back; ?>"><i class="fa-solid fa-arrow-left-long me-1"></i> Dashboard</a>
    </div>
  </div>

  <!-- Table -->
  <div class="card shadow-sm">
    <div class="table-responsive">
      <table id="lettersTable" class="table table-striped align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th>Letter No.</th>
            <th>Date</th>
            <th>Department</th>
            <th>Subject</th>
            <th>Received Date</th>
            <th>Scan</th>
            <th>Action</th>
            <th>Description</th>
            <th>Branch</th>
          </tr>
        </thead>
        <tbody>
        <?php if ($result && $result->num_rows > 0): ?>
          <?php while ($row = $result->fetch_assoc()): ?>
            <tr>
              <td><?= htmlspecialchars($row['letter_number'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['letter_date'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['department'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['subject'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= !empty($row['received_date']) ? htmlspecialchars($row['received_date'], ENT_QUOTES, 'UTF-8') : '—'; ?></td>
              <td>
                <?php if (!empty($row['scan_copy'])): ?>
                  <a class="btn btn-sm btn-outline-primary"
                     href="<?= htmlspecialchars($row['scan_copy'], ENT_QUOTES, 'UTF-8'); ?>"
                     target="_blank"><i class="fa-solid fa-up-right-from-square me-1"></i>View</a>
                <?php else: ?>
                  <span class="text-muted">N/A</span>
                <?php endif; ?>
              </td>
              <td><?= htmlspecialchars($row['action'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['description'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></td>
              <td><?= htmlspecialchars($row['branch_name'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
            </tr>
          <?php endwhile; ?>
        <?php else: ?>
          <tr><td colspan="9" class="text-center py-4">No letters found.</td></tr>
        <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<!-- JS: jQuery + DataTables + Bootstrap 5 theme -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
<script>
  $(function(){
    $('#lettersTable').DataTable({
      pageLength: 25,
      order: [[1, 'desc']], // by Date desc
      // You can enable horizontal scroll if you anticipate long text:
      // scrollX: true
    });
  });
</script>
</body>
</html>
