<?php
// Start the session and include the database connection
session_start();
include 'db_connection.php';  // Include your database connection file

// Check if the user is logged in and is a Branch Head
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'branch_head') {
    header("Location: login.php");  // Redirect to login page if not logged in as branch_head
    exit();
}

// Get the branch_id of the logged-in user (Branch Head)
$branch_id = $_SESSION['branch_id'];  // Assuming branch_id is stored in session

// Fetch the letters that are in "received_assigned" status for the logged-in Branch Head, including the username for forward_to field
$query = "SELECT letters.*, users.username AS forward_to_user 
          FROM letters 
          LEFT JOIN users ON letters.forward_to = users.user_id 
          WHERE letters.branch_id = ? AND letters.status_branch_head = 'received_assigned' 
          ORDER BY letters.letter_date DESC";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $branch_id);  // Bind the branch_id parameter
$stmt->execute();
$result = $stmt->get_result();

// Check for errors
if ($result === false) {
    die("Error fetching letters: " . $conn->error);
}

// Close the database connection
$stmt->close();
$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Branch Head - Received Assigned Letters</title>

    <!-- Bootstrap CSS -->
    <link rel="stylesheet" href="https://maxcdn.bootstrapcdn.com/bootstrap/4.0.0/css/bootstrap.min.css">

    <!-- DataTables CSS -->
    <link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.10.21/css/jquery.dataTables.css">
</head>
<body>
    <div class="container">
        <h2 class="mt-5">Received and Assigned Letters (Branch Head)</h2>
        
        <!-- If no letters found -->
        <?php if ($result->num_rows === 0): ?>
            <div class="alert alert-info">No received and assigned letters found for your branch.</div>
        <?php else: ?>
            <!-- Letters table -->
            <table id="lettersTable" class="table table-striped mt-4">
                <thead>
                    <tr>
                        <th>Letter Number</th>
                        <th>Letter Date</th>
                        <th>Department</th>
                        <th>Subject</th>
                        <th>Forward To</th> <!-- Display the forward_to username -->
                        <th>Received Date</th>
                        <th>Scan Copy</th>
                    </tr>
                </thead>
                <tbody>
                    <?php while ($row = $result->fetch_assoc()): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($row['letter_number']); ?></td>
                            <td><?php echo htmlspecialchars($row['letter_date']); ?></td>
                            <td><?php echo htmlspecialchars($row['department']); ?></td>
                            <td><?php echo htmlspecialchars($row['subject']); ?></td>
                            <td><?php echo $row['forward_to_user'] ? htmlspecialchars($row['forward_to_user']) : 'Not Assigned'; ?></td> <!-- Display the forward_to user -->
                            <td><?php echo $row['received_date'] ? htmlspecialchars($row['received_date']) : 'N/A'; ?></td>
                            <td>
                                <?php if (!empty($row['scan_copy'])): ?>
                                    <a href="<?php echo htmlspecialchars($row['scan_copy']); ?>" target="_blank">View Scan Copy</a>
                                <?php else: ?>
                                    No Scan Copy Available
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
        <?php endif; ?>
        
        <!-- Button to go back to the dashboard -->
        <a href="branch_head_dashboard.php" class="btn btn-primary">Back to Dashboard</a>
    </div>

    <!-- Bootstrap JS and dependencies -->
    <script src="https://code.jquery.com/jquery-3.2.1.slim.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/popper.js/1.12.9/umd/popper.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.0.0/js/bootstrap.min.js"></script>

    <!-- DataTables JS -->
    <script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.21/js/jquery.dataTables.js"></script>

    <script>
        $(document).ready(function() {
            // Initialize DataTable
            $('#lettersTable').DataTable({
                "paging": true,         // Enable pagination
                "searching": true,      // Enable searching
                "ordering": true,       // Enable sorting
                "info": true            // Show table info
            });
        });
    </script>
</body>
</html>
