<?php
// Start the session and include the database connection
session_start();
require_once 'db_connection.php';  // Include your database connection file

// Check if the user is logged in and is a Staff User (not Branch Head)
if (!isset($_SESSION['user_id']) || ($_SESSION['role'] ?? '') !== 'staff_user') {
    header("Location: login.php");
    exit();
}

// Get the user_id of the logged-in user (Staff User)
$user_id = (int)($_SESSION['user_id'] ?? 0);

// Fetch the letters where action is 'Yes' and description is not empty, join file name
$query = "
    SELECT l.letter_id, l.letter_number, l.letter_date, l.department, l.subject,
           l.received_date, l.scan_copy, l.action, l.description, fm.file_name
    FROM letters l
    LEFT JOIN file_management fm ON l.file_id = fm.file_id
    WHERE l.forward_to = ?
      AND l.action = 'Yes'
      AND l.description IS NOT NULL
      AND l.description <> ''
    ORDER BY l.letter_date DESC
";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$res = $stmt->get_result();

$letters = [];
while ($row = $res->fetch_assoc()) { $letters[] = $row; }

$stmt->close();
$conn->close();

function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
?>
<!DOCTYPE html>
<html lang="en" data-bs-theme="light">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Actioned Letters • Staff User</title>

  <!-- Bootstrap 5.3 & Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
  <!-- Simple-DataTables (no jQuery) -->
  <link href="https://cdn.jsdelivr.net/npm/simple-datatables@9.0.0/dist/style.css" rel="stylesheet">

  <style>
    :root{
      --brand:#155eef; --border:#e8ecf7; --surface:#ffffff; --radius:14px;
    }
    body{ background:#f6f8ff; }
    .page-wrap{ max-width: 1280px; }
    .page-card{
      background:var(--surface);
      border:1px solid var(--border);
      border-radius:var(--radius);
    }
    thead th{
      background:#f8faff; position:sticky; top:0; z-index:1;
      border-bottom:1px solid var(--border);
    }
    .scan-link{
      max-width:240px; display:inline-block; white-space:nowrap; overflow:hidden; text-overflow:ellipsis;
    }
    .badge-soft-success{ background:#e7f6ec; color:#1b7a38; border:1px solid #cfead8; }
    .desc-cell{ max-width: 380px; }
    .datatable-wrapper .datatable-top{ padding:0 0 1rem 0; }
    .datatable-wrapper .datatable-bottom{ padding:1rem 0 0 0; }
  </style>
</head>
<body>

  <!-- Header -->
  <header class="border-bottom bg-white">
    <div class="container page-wrap d-flex align-items-center justify-content-between py-3">
      <div class="d-flex align-items-center gap-2">
        <a href="staff_user_dashboard.php" class="btn btn-outline-secondary btn-sm"><i class="bi bi-arrow-left"></i></a>
        <div>
          <h5 class="mb-0">Letters with Action “Yes”</h5>
          <small class="text-secondary">Descriptions provided • Your assignments</small>
        </div>
      </div>
      <img src="logo.jpg" alt="Logo" width="44" height="44" class="rounded">
    </div>
  </header>

  <main class="container page-wrap my-4">
    <div class="page-card p-3 p-md-4">
      <div class="d-flex align-items-center justify-content-between mb-3">
        <div class="d-flex align-items-center gap-2">
          <span class="badge text-bg-primary"><i class="bi bi-check2-square me-1"></i> Actioned</span>
          <span class="text-secondary small">Total: <strong><?= count($letters) ?></strong></span>
        </div>
        <div class="d-flex gap-2">
          <a href="view_letters_statistics.php" class="btn btn-outline-primary btn-sm">
            <i class="bi bi-pie-chart me-1"></i> Statistics
          </a>
        </div>
      </div>

      <?php if (!count($letters)): ?>
        <div class="text-center py-5">
          <i class="bi bi-inboxes fs-1 text-secondary d-block mb-2"></i>
          <h6 class="mb-1">No actioned letters found</h6>
          <p class="text-secondary mb-3">There are no letters with Action = Yes and a description.</p>
          <a href="staff_user_dashboard.php" class="btn btn-primary btn-sm"><i class="bi bi-arrow-left-short"></i> Back to Dashboard</a>
        </div>
      <?php else: ?>
        <div class="table-responsive">
          <table id="lettersTable" class="table table-hover align-middle mb-0">
            <thead>
              <tr>
                <th>Letter No.</th>
                <th>Letter Date</th>
                <th>Department</th>
                <th>Subject</th>
                <th>Received</th>
                <th>Scan</th>
                <th>Action</th>
                <th>Description</th>
                <th>File Name</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($letters as $row): ?>
                <tr>
                  <td><?= h($row['letter_number']) ?></td>
                  <td><?= h($row['letter_date']) ?></td>
                  <td><?= h($row['department']) ?></td>
                  <td class="text-truncate" style="max-width:360px;" title="<?= h($row['subject']) ?>"><?= h($row['subject']) ?></td>
                  <td><?= $row['received_date'] ? h($row['received_date']) : 'N/A' ?></td>
                  <td>
                    <?php if (!empty($row['scan_copy'])): ?>
                      <a class="scan-link" href="<?= h($row['scan_copy']) ?>" target="_blank" rel="noopener">
                        <i class="bi bi-box-arrow-up-right me-1"></i>View
                      </a>
                    <?php else: ?>
                      <span class="text-secondary">No file</span>
                    <?php endif; ?>
                  </td>
                  <td><span class="badge badge-soft-success">Yes</span></td>
                  <td class="desc-cell">
                    <div class="text-truncate" title="<?= h($row['description']) ?>"><?= h($row['description']) ?></div>
                  </td>
                  <td><?= $row['file_name'] ? h($row['file_name']) : '<span class="text-secondary">No File Associated</span>' ?></td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>

        <div class="text-center mt-4">
          <a href="staff_user_dashboard.php" class="btn btn-primary btn-lg"><i class="bi bi-grid me-1"></i> Back to Dashboard</a>
        </div>
      <?php endif; ?>
    </div>
  </main>

  <!-- JS -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/simple-datatables@9.0.0" ></script>
  <script>
    (function(){
      const tbl = document.getElementById('lettersTable');
      if (!tbl) return;
      new simpleDatatables.DataTable(tbl, {
        searchable: true,
        fixedHeight: false,
        perPage: 10,
        perPageSelect: [10, 20, 50, 100],
        labels: {
          placeholder: "Search...",
          perPage: "{select} per page",
          noRows: "No entries to show",
          info: "Showing {start}–{end} of {rows} entries"
        }
      });
    })();
  </script>
</body>
</html>
